﻿(function ($) {
    var listViewConstants = {
        saveStateWaitInterval: 1000
    }

    $.widget("a4.a4listview", {
        options: {
            expanded: false,
            onDemand: false,
            stateKey: undefined,
            actionButtons: [],
            localizableTexts: [],
            actionsButtonsDropDown: false,
            displayActionButtonsOnHover: true,
            displayActionButtonsOnLeft: false,
            ajaxAction: undefined,
            ajaxParams: {},
            customItems: undefined,
            icons: {
                Root: undefined,
                Node: { Icon: "folder", SelectedIcon: "folder-open" },
                Leaf: undefined,
                Custom: []
            },
            properties: {
                Key: "Id",
                Id: "Id",
                ParentId: "ParentId",
                Children: "Children",
                Label: "Name",
                HasChildren: "HasChildren",
                Selected: "ListView_Internal_Selected"
            },
            displaySearchBox: true,
            altValueField: undefined, //Input element that is to be updated with the selected values from the listview.            
            height: 'full',
            serverSide: true,
            data: [],
            highlightFirstNode: false,
            allowNodeHighlight: true,
            allowNodeMultiHighlight: false,
            allowNodeSelect: false,
            showTooltip: false
        },
        _create: function () {
            var self = this;
            var loadStateCall = null;

            if (this.options.stateKey) {
                loadStateCall = this._loadState();
            }
            else {
                this.state = { Nodes: [] };
            }

            this.selectedNodes = [];
            this.addedNodes = [];

            $.when(loadStateCall).then(function () {
                self._render();
                self._loadData();
                self._bindEvents();
            });
        },
        destroy: function () {
            $.Widget.prototype.destroy.call(this);
        },
        //------------------------------
        _loadState: function () {
            var self = this;
            return a4.callServerMethod(a4.getAction("GetListViewConfiguration", "Shared", "Home"), { type: this.options.stateKey, itemId: 0 }, function (result) {
                if (result && result.Nodes)
                    self.state = { Nodes: _.filter(result.Nodes, function (s) { return s.Key != null; }) };
                else
                    self.state = { Nodes: [] };

                self._trigger("loadState", null, { state: self.state });
            });
        },
        _saveState: _.debounce(function () {
            if (this.options.stateKey) {
                a4.callServerMethod(a4.getAction("UpdateListViewConfiguration", "Shared", "Home"), { type: this.options.stateKey, itemId: 0, config: this.state }, function (result) { });
            }
        }, listViewConstants.saveStateWaitInterval),
        //------------------------------
        setData: function (data) {
            this.options.data = data;
        },
        setAjaxParams: function (params) {
            this.options.ajaxParams = _.extend(this.options.ajaxParams, params);
        },
        refresh: function (highlightedNodeKey) {
            this._loadData(highlightedNodeKey);
        },
        evaluateActionsConditions: function (nodeKey) {
            var self = this;
            var listItems = nodeKey ? $(".node[data-nodekey='" + nodeKey + "']", self.element) : $(".node", self.element);

            listItems.each(function () {
                var listItem = $(this);
                var nodeData = self._getNodeData(listItem.attr("data-nodekey"));

                _.each(self.options.actionButtons, function (b) {
                    var actionButton = $(".h-actionButton[data-position='" + b.Position + "']", listItem);

                    if (actionButton.length > 0)
                        actionButton.toggleClass("hidden", !a4.evaluateCondition(b.DisplayCondition, nodeData));
                });
            });
        },
        //------------------------------
        addNode: function (node) {
            this.addedNodes.push(node);
            this.refresh();
        },
        getNodes: function () {
            return this.data.Nodes;
        },
        //------------------------------
        getHighlightedNodes: function () {
            var self = this;
            var nodes;

            if (this.state.Nodes.length > 0) {
                var nodesState = _.filter(this.state.Nodes, function (n) { return n.Highlighted == true });

                if (nodesState && nodesState.length > 0) {
                    nodes = _.map(nodesState, function (n) { return self._getNodeData(n.Key); });
                }
            }

            return nodes;
        },
        getHighlightedNode: function () {
            return this.lastHighlightedNode;
        },
        highlightNode: function (nodeKey) {
            this._highlightNode(this._getNodeData(nodeKey), false);
        },
        _highlightNode: function (node, multiple) {
            if (this.options.allowNodeHighlight) {
                var self = this;
                var saveState = false;

                if (node) {
                    var deferred = $.Deferred();
                    var promise;

                    if (!self._trigger("beforeHighlightNode", null, { node: self.lastHighlightedNode, isCustom: node.ListView_Internal_IsCustom, deferred: deferred }))
                        promise = deferred.promise();

                    $.when(promise).then(function () {
                        if (self.options.allowNodeMultiHighlight) {
                            if (multiple && self.lastHighlightedNode && self.lastHighlightedNode.Element) {
                                var lastIndex = $("li.node", self.element).index(self.lastHighlightedNode.Element);
                                var nodeIndex = $("li.node", self.element).index(node.Element);

                                for (var index = Math.min(lastIndex, nodeIndex); index <= Math.max(lastIndex, nodeIndex); index++) {
                                    var element = $("li.node", self.element).get(index);
                                    var nodeKey = $(element).attr("data-nodekey");
                                    self._toggleNode(self._getNodeData(nodeKey), true);
                                }
                            }
                            else {
                                self._toggleNode(node);
                            }

                            saveState = true;
                        }
                        else {
                            var highlightedNode = self.lastHighlightedNode;
                            var highlightedNodeKey;

                            if (highlightedNode)
                                highlightedNodeKey = highlightedNode.ListView_Internal_Key;

                            if (node.ListView_Internal_Key != highlightedNodeKey) {

                                //Unhighlight highlighted node
                                self._toggleNode(highlightedNode, false);

                                //Highlight node
                                self._toggleNode(node, true);

                                saveState = true;
                            }
                        }

                        //Get highlighted nodes
                        if (self.options.altValueField && !self.options.allowNodeSelect) {
                            var highlightedNodeKeys = _.pluck(self.getHighlightedNodes(), "ListView_Internal_Key");
                            self.options.altValueField.val(highlightedNodeKeys.join());
                        }

                        if (saveState) {
                            self._saveState();
                        }
                    });
                }
            }
        },
        _toggleNode: function (node, isHighlighted) {
            if (node) {
                var stateNode = this._getNodeState(node.ListView_Internal_Key);

                if (stateNode) {
                    if (isHighlighted == undefined) {
                        isHighlighted = !stateNode.Highlighted;
                    }

                    stateNode.Highlighted = isHighlighted;

                    stateNode.Parent = node.ListView_Internal_ParentKey;
                }

                if (node.Element && node.Element.length > 0) {
                    node.Element.toggleClass("highlighted", isHighlighted);
                }

                if (isHighlighted)
                    this.lastHighlightedNode = node;

                this._trigger("selectNode", null, { node: node, selected: isHighlighted, isCustom: node.ListView_Internal_IsCustom });
            }
        },
        _completeToggle: function (nodeElement, key) {
            var self = this;
            nodeElement.toggleClass("expanded");
            nodeElement.next("ul.sub-menu").toggle();
            this._saveState();
        },
        //------------------------------        
        getSelectedNodes: function () {
            if (this.options.allowNodeSelect) {
                var selectedProperty = this.options.properties["Selected"];
                return _.filter(this.data.Nodes, function (n) { return n[selectedProperty] == true; });
            }

            return [];
        },
        selectNodes: function (nodeKeys) {
            var self = this;
            _.each(nodeKeys, function (nodeKey) {
                self.selectNode(nodeKey);
            });
        },
        selectNode: function (nodeKey, partial) {
            if (partial)
                this._selectNode(this._partialSelectNode(nodeKey));
            else
                this._selectNode(this._getNodeData(nodeKey));
        },
        _selectNode: function (node) {
            if (node && node.Element && node.Element.length > 0 && this.options.allowNodeSelect) {
                var selectedProperty = this.options.properties["Selected"];
                var isSelected = node[selectedProperty] == true;

                node.Element.toggleClass("selected", !isSelected);

                node[selectedProperty] = !node[selectedProperty];

                this._updateRelatedSelectedNodes();

                if (this.options.altValueField) {
                    var selectedNodeKeys = _.pluck(this.getSelectedNodes(), "ListView_Internal_Key");
                    this.options.altValueField.val(selectedNodeKeys.join());
                }
            }
        },
        _updateRelatedSelectedNodes: function () {
            var self = this;
            var selectedNodes = this.getSelectedNodes();

            $(".node", this.element).removeClass("partial-selected disable-select");

            _.each(selectedNodes, function (node) {
                var parentNodes = self._getParentNodes(node);

                if (parentNodes && parentNodes.length > 0) {
                    _.each(parentNodes, function (parentNode) {
                        parentNode.Element.addClass("partial-selected");
                    });
                }

                var children = self._getChildrenNodesRecursive(node);

                if (children && children.length > 0) {
                    _.each(children, function (child) {
                        if (child && child.Element && child.Element.length > 0)
                            child.Element.addClass("disable-select");
                    });
                }
            });
        },
        _partialSelectNode: function (node) {
            if (node && node.Element && node.Element.length > 0)
                node.Element.toggleClass("partial-selected");
        },
        //------------------------------                
        _renderListView: function (data, highlightedNodeKey) {
            var self = this;

            //Flatten data and initialize internal cache
            self._initializeNodeData(data, highlightedNodeKey);

            var highlightedNodeState = _.find(self.state.Nodes, function (n) { return n.Highlighted == true; });

            if (!self.lastHighlightedNode && highlightedNodeState) {
                self.lastHighlightedNode = self._getNodeData(highlightedNodeState.Key);

                while (highlightedNodeState && !self.lastHighlightedNode) {
                    highlightedNodeState = _.find(self.state.Nodes, function (n) { return n.Key == highlightedNodeState.Parent; });

                    if (highlightedNodeState) {
                        self.lastHighlightedNode = self._getNodeData(highlightedNodeState.Key);
                    }
                }
            }

            if (!self.lastHighlightedNode && self.options.highlightFirstNode) {
                self.lastHighlightedNode = self.data.Nodes[0];
            }

            //Updating state with last highlighted node
            if (self.lastHighlightedNode) {
                //Deselect previously highlighted node
                if (highlightedNodeState)
                    highlightedNodeState.Highlighted = false;

                //Highlight new node in state
                highlightedNodeState = self._getNodeState(self.lastHighlightedNode.ListView_Internal_Key);

                if (!highlightedNodeState) {
                    highlightedNodeState = { Key: self.lastHighlightedNode.ListView_Internal_Key, Parent: self.lastHighlightedNode.ListView_Internal_ParentKey, Expanded: false, Highlighted: true };
                    self.state.Nodes.push(highlightedNodeState);
                }
                else {
                    highlightedNodeState.Highlighted = true;
                }
            }

            //Update state to remove deleted nodes
            self.state.Nodes = _.filter(self.state.Nodes, function (n) {
                return _.some(self.data.Nodes, function (x) {
                    return x.ListView_Internal_Key == n.Key;
                });
            });

            //Include addedNodes
            for (var index in self.addedNodes) {
                self.addedNodes[index].Highlighted = false;
                self.data.Nodes.push(self.addedNodes[index]);

                for (var i in self.data.Nodes) {
                    if (self.data.Nodes[i].Name == self.addedNodes[index].ListView_Internal_ParentKey) {
                        self.data.Nodes[i].Choices.push(self.addedNodes[index]);
                        break;
                    }
                }
            }

            self._renderList(self.data.Nodes);

            var searchValue = $(".h-searchBox", self.element).val();

            if (searchValue)
                self._search(searchValue);

            self._trigger("draw", null, { count: self.data.Nodes.length, selectedNode: self.lastHighlightedNode, isCustom: self.lastHighlightedNode ? self.lastHighlightedNode.ListView_Internal_IsCustom : false });
        },
        _loadData: function (highlightedNodeKey) {
            var self = this;
            var promise = null;

            if (self.options.serverSide)
                promise = a4.callServerMethod(self.options.ajaxAction, self.options.ajaxParams);
            else
                promise = self.options.data;

            return $.when(promise).then(function (data) {
                if (data) {
                    if (data.d) {
                        data = _.isString(data.d) ? JSON.parse(data.d) : data.d;
                    }

                    self._renderListView(data, highlightedNodeKey);
                }
            });
        },
        _initializeNodeData: function (nodes, highlightedNodeKey) {
            var self = this;

            if (!this.data || !highlightedNodeKey) {
                this.data = { Nodes: [] };
            }

            this.lastHighlightedNode = null;

            if (nodes && nodes[0] && typeof nodes[0][self.options.properties.Children] != 'undefined') {
                //Flatten the nodes to store it in internal data
                this.data.Nodes = this.data.Nodes.concat(this._flattenHierarchicalNodeData(nodes, highlightedNodeKey));
            }
            else {
                this.data.Nodes = this.data.Nodes.concat(nodes);

                //Populating internal node key property
                _.each(this.data.Nodes, function (n) {
                    self._setNodeInternalKey(n);
                });
            }

            //Add custom items
            if (this.options.customItems && _.isArray(this.options.customItems)) {
                _.each(this.options.customItems, function (n) {
                    n.ListView_Internal_Key = _.uniqueId("custom_");
                    n.ListView_Internal_IsCustom = true;
                });

                self.data.Nodes = _.union(this.options.customItems, self.data.Nodes);
            }
        },
        _flattenHierarchicalNodeData: function (nodes, highlightedNodeKey) {
            if (!nodes || nodes.length == 0)
                return null;

            var self = this;

            _.each(nodes, function (n) {
                self._setNodeInternalKey(n, true);

                if (_.size(n[self.options.properties.Children]) > 0) {
                    _.each(n[self.options.properties.Children], function (c) {
                        c.ListView_Internal_ParentKey = n.ListView_Internal_Key;
                    });
                }
                else if (highlightedNodeKey) {
                    n.ListView_Internal_ParentKey = highlightedNodeKey;
                }
            });

            return _.compact(_.flatten(_.union(nodes, _.map(nodes, function (n) {
                return self._flattenHierarchicalNodeData(n[self.options.properties.Children]);
            }))));
        },
        _setNodeInternalKey: function (node, nested) {
            if (node && !node.ListView_Internal_Key) {
                var self = this;
                var keyProperty = this.options.properties.Key;
                var key;

                if (!nested && node.ListView_Internal_ParentKey == null && node[self.options.properties.ParentId] != null) {
                    var parentNode = _.find(self.data.Nodes, function (p) { return p[self.options.properties.Id] == node[self.options.properties.ParentId]; });

                    if (parentNode && parentNode[self.options.properties.Id] != node[self.options.properties.Id]) {
                        this._setNodeInternalKey(parentNode);
                        node.ListView_Internal_ParentKey = parentNode.ListView_Internal_Key;
                    }
                }

                if (_.isArray(keyProperty)) {
                    key = _.compact(_.map(keyProperty, function (k) { return node[k] ? node[k].toString() : ""; })).join("_");
                }
                else {
                    key = node[keyProperty];
                }

                if (node.ListView_Internal_ParentKey) {
                    key = node.ListView_Internal_ParentKey + "_" + key;
                }

                node.ListView_Internal_Key = key;
            }
        },
        _getNodeState: function (key) {
            var nodeState = _.find(this.state.Nodes, function (n) { return n.Key == key; });
            return nodeState;
        },
        _getNodeData: function (key) {
            return _.find(this.data.Nodes, function (n) { return n.ListView_Internal_Key == key; });
        },
        _getRootNodes: function (nodes) {
            var self = this;
            var rootNodes;

            if (nodes && nodes.length > 0) {
                rootNodes = _.filter(nodes, function (n) { return !n.ListView_Internal_ParentKey || !_.some(nodes, function (x) { return n.ListView_Internal_ParentKey == x.ListView_Internal_Key; }); });
            }

            return rootNodes;
        },
        _getParentNode: function (node) {
            var parentNode;

            if (node.ListView_Internal_ParentKey) {
                parentNode = _.find(this.data.Nodes, function (n) { return n.ListView_Internal_Key == node.ListView_Internal_ParentKey; });
            }

            return parentNode;
        },
        _getParentNodes: function (node) {
            var parent = this._getParentNode(node);

            if (parent) {
                return _.union([parent], this._getParentNodes(parent));
            }

            return null;
        },
        _hasChildren: function (node) {
            return ((node.Children && node.Children.length > 0) ||
                _.any(this.data.Nodes, function (n) { return n.ListView_Internal_ParentKey == node.ListView_Internal_Key; }));
        },
        _getChildrenNodesRecursive: function (node) {
            var self = this;
            var children = this._getChildrenNodes(this.data.Nodes, node);

            if (children && children.length > 0) {
                var allChildren = [];

                _.each(children, function (child) {
                    allChildren = _.union(allChildren, [child], self._getChildrenNodesRecursive(child));
                });

                return allChildren;
            }

            return null;
        },
        _getChildrenNodes: function (nodes, node) {
            var self = this;
            var childrenNodes;

            if (nodes && nodes.length > 0) {
                childrenNodes = _.filter(nodes, function (n) { return n.ListView_Internal_ParentKey && node.ListView_Internal_Key == n.ListView_Internal_ParentKey; });
            }

            return childrenNodes;
        },
        _render: function () {
            var container = $("<div />", { "class": "list-view" });

            if (this.options.displaySearchBox && !this.options.onDemand) {
                var searchBox = $("<div />", { "class": "search-box" });

                searchBox.append($("<input />", { "type": "text", "class": "h-searchBox", "placeholder": this._getLocalizableText("Search") }));
                searchBox.append(voxco.icons.getIcon("search", "h-searchButton search-button"));

                container.append(searchBox);
            }

            container.append($("<div />", { "class": "list-view-message" }).append(voxco.icons.getIcon("loading")));

            var content = $("<div />", { "class": "list-view-content" });

            content.hide();

            container.append(content);

            this.element.html(container);
        },
        _renderList: function (nodes, expanded) {
            var self = this;

            var listView = $(".list-view-content", this.element);

            listView.empty();

            if (nodes.length > 0) {
                var rootNodes = self._getRootNodes(nodes);

                if (rootNodes && rootNodes.length != nodes.length) {
                    _.each(rootNodes, function (n) {
                        var subMenu = $("<ul />", { "class": "sub-menu" });
                        self._renderNode(subMenu, nodes, n, 0, expanded);
                        listView.append(subMenu);
                    });
                }
                else {
                    var subMenu = $("<ul />", { "class": "sub-menu" });

                    _.each(rootNodes, function (n) {
                        self._renderNode(subMenu, nodes, n, 0, expanded);
                    });

                    listView.append(subMenu);
                }
            }
            else {
                listView.append($("<ul />", { "class": "sub-menu" }).append($("<li />", { "class": "node message" }).append(dataTableResources.EmptyTable)));
            }

            if (this.options.height && this.options.height != "full") {
                listView.height(this.options.height);
            }

            listView.show();
            $(".list-view-message", this.element).hide();

            this._updateRelatedSelectedNodes();

            this._adjustHeight();
        },
        _getNodeLabel: function (node, type) {
            var label;
            var labelProperty = this.options.properties[type] || this.options.properties["Node"];

            if (labelProperty && _.isObject(labelProperty)) {
                label = node[labelProperty.Label];
            }

            if (!label) {
                label = node[this.options.properties.Label];
            }

            return label;
        },
        _renderNode: function (parent, nodes, node, depth, expanded) {
            var self = this;
            var children = this._getChildrenNodes(nodes, node);
            var isExpanded = expanded;
            var hasOnDemandChildren = (this.options.onDemand && node[this.options.properties.HasChildren]);

            var isRoot = depth == 0;
            var isLeaf = (!children || children.length == 0) && !hasOnDemandChildren;

            var nodeState = this._getNodeState(node.ListView_Internal_Key);

            if (!nodeState) {
                nodeState = { Key: node.ListView_Internal_Key, Parent: node.ListView_Internal_ParentKey, Expanded: false, Highlighted: false };
                this.state.Nodes.push(nodeState);
            }

            if (isExpanded == undefined)
                isExpanded = nodeState.Expanded || this.options.expanded;

            var listItemClass = "node";
            var nodeType = "Node";

            if (isRoot) {
                listItemClass += " root";
                nodeType = "Root";
            }
            else if (isLeaf) {
                listItemClass += " leaf";
                nodeType = "Leaf";
            }

            if (isExpanded) {
                listItemClass += " expanded";
            }

            if (nodeState.Highlighted) {
                listItemClass += " highlighted";
            }

            if (node.ListView_Internal_IsCustom) {
                listItemClass += " custom";
            }

            var listItem = $("<li />", { "class": listItemClass, "data-nodekey": node.ListView_Internal_Key });

            var padding = (depth * 8) + 5;

            if (children.length > 0 || hasOnDemandChildren) {
                listItem.append(voxco.icons.getIcon("caret-down", "h-toggle toggle expanded", { FixedWidth: true }));
                listItem.append(voxco.icons.getIcon("caret-right", "h-toggle toggle collapsed", { FixedWidth: true }));

                // Check if we have already loaded the children of this parent
                if (hasOnDemandChildren && !this._hasChildren(node)) {
                    listItem.addClass("lazy");
                }
            }
            else if (depth > 0) {
                padding += 17;
            }

            if (this.options.allowNodeSelect) {
                var selectNodeButton = $("<span />", { "class": "fa fa-fw h-select select-node" });

                var selectedProperty = this.options.properties["Selected"];

                if (selectedProperty && node[selectedProperty] == true)
                    listItem.addClass("selected");

                listItem.append(selectNodeButton);
            }

            var icons = this.options.icons;

            if (icons) {
                var icon;
                var selectedIcon;

                if (icons.Custom && _.isArray(icons.Custom) && icons.Custom.length > 0) {
                    icon = _.find(icons.Custom, function (i) { return a4.evaluateCondition(i.Condition, node); });
                }

                if (!icon) {
                    icon = icons[nodeType] || icons["Node"];
                }

                if (icon && _.isObject(icon)) {
                    selectedIcon = icon.SelectedIcon;
                    icon = icon.Icon;
                }

                if (selectedIcon) {
                    listItem.append(voxco.icons.getIcon(icon, "node-icon default v-nodeIcon"));
                    listItem.append(voxco.icons.getIcon(selectedIcon, "node-icon highlighted v-nodeIcon"));
                }
                else if (icon) {
                    listItem.append(voxco.icons.getIcon(icon, "node-icon v-nodeIcon"));
                }
            }

            var textSpan = $("<span />", { "class": "text" });
            var labelText = this._getNodeLabel(node, nodeType);
            if (this.options.showTooltip) {
                textSpan = $("<span />", { "class": "text", "title": labelText });
            }
            textSpan.append(labelText);

            if (this.options.displayActionButtonsOnLeft)
                this._renderActionButtons(node, listItem);

            listItem.append(textSpan);

            if (!this.options.displayActionButtonsOnLeft)
                this._renderActionButtons(node, listItem);

            listItem.css("padding-left", padding);

            parent.append(listItem);

            //Setting jquery element to node data
            node.Element = listItem;

            if (children.length > 0 || hasOnDemandChildren) {
                var subMenu = $("<ul />", { "class": "sub-menu" });

                subMenu.css("display", isExpanded ? "block" : "none");

                _.each(children, function (n) {
                    subMenu.append(self._renderNode(subMenu, nodes, n, depth + 1, expanded));
                });

                parent.append(subMenu);
            }
        },
        _renderActionButtons: function (node, listItem) {
            var self = this;

            if (this.options.actionButtons.length > 0) {
                var actionButtonsDiv = $("<div />", { "class": "h-actionButtons action-buttons" });

                if (this.options.displayActionButtonsOnLeft)
                    actionButtonsDiv.addClass("left");
                else
                    actionButtonsDiv.addClass("right");

                if (this.options.displayActionButtonsOnHover)
                    actionButtonsDiv.addClass("display-on-hover");

                if (this.options.actionsButtonsDropDown) {
                    var buttonGroup = $("<div />", { "class": "button-group dropdown-container" });

                    var button = $("<div />", { "class": "button icon-only dropdown-toggle" });
                    button.append(voxco.icons.getIcon("gears"));
                    button.append(voxco.icons.getIcon("caret-down", "icon-right"));

                    var dropDownMenu = $("<ul />", { "class": "dropdown-menu" });

                    _.each(this.options.actionButtons, function (b) {
                        var actionButton = $("<li />", { "class": "action-button h-actionButton", "data-action": b.Action }).append($("<a />", { "href": "#" }).append(b.Label));

                        actionButton.toggleClass("hidden", !a4.evaluateCondition(b.DisplayCondition, node))

                        dropDownMenu.append(actionButton);
                    });

                    buttonGroup.append(button);

                    buttonGroup.append(dropDownMenu);

                    actionButtonsDiv.append(buttonGroup);
                }
                else {
                    _.each(this.options.actionButtons, function (b, i) {
                        b.Position = i;

                        var actionButton = $("<span />", { "class": "action-button h-actionButton", "title": b.Label, "data-action": b.Action, "data-position": b.Position }).append(voxco.icons.getIcon(b.Icon, "", { FixedWidth: true }));

                        actionButton.toggleClass("hidden", !a4.evaluateCondition(b.DisplayCondition, node))

                        actionButtonsDiv.append(actionButton);
                    });
                }

                listItem.append(actionButtonsDiv);
            }
        },
        _search: function (value) {
            var self = this;
            var labelProperty = this.options.properties.Label;
            var nodes;
            var expanded;

            if (value && value.length > 0) {
                nodes = [];

                var matcher = new RegExp($.ui.autocomplete.escapeRegex(value), "i");

                var filteredNodes = _.filter(this.data.Nodes, function (n) { return matcher.test(n[labelProperty]) });

                if (filteredNodes.length > 0) {
                    nodes = filteredNodes;
                    expanded = true;

                    if (this.data.Nodes.length > filteredNodes.length) {
                        //Load parents for each node
                        _.each(filteredNodes, function (n) {
                            nodes = _.union(nodes, self._getParentNodes(n));
                        });

                        nodes = _.uniq(nodes, false, function (f) { return f.ListView_Internal_Key; });
                    }
                }
            }
            else {
                nodes = this.data.Nodes;
            }

            this._renderList(nodes, expanded);
        },
        _bindEvents: function () {
            var self = this;

            if (this.options.height == "full") {
                $(window).on("resize", function () {
                    //var listView = $(".list-view-content", this.element);
                    //listView.height($(window).height() - listView.offset().top/* - 20*/);

                    self._adjustHeight();
                });
            }

            this._on({
                "click span.h-toggle": function (event) {
                    var element = $(event.currentTarget);
                    var nodeElement = element.closest("li.node");
                    var key = nodeElement.attr("data-nodekey");

                    //Updating state - before lazy load so that initialDataLoad has the proper state
                    var nodeState = self._getNodeState(key);
                    nodeState.Expanded = !nodeState.Expanded;

                    if (!nodeElement.hasClass("lazy")) {
                        self._completeToggle(nodeElement, key);
                    }
                    else {
                        var nodeInfo = self._getNodeData(key);
                        var listViewKey = this.options.properties.Key;

                        if (!_.isArray(listViewKey)) {
                            self.options.ajaxParams[listViewKey] = nodeInfo[listViewKey];
                        }
                        else {
                            _.each(listViewKey, function (k) {
                                self.options.ajaxParams[k] = nodeInfo[k];
                            });
                        }

                        $.when(self._loadData(key)).then(function () {
                            nodeElement.removeClass("lazy")
                            self._completeToggle(nodeElement, key);
                        });
                    }

                    event.stopPropagation();
                },
                "click span.h-select": function (event) {
                    var element = $(event.currentTarget);
                    var nodeElement = element.closest("li.node");
                    var nodeKey = nodeElement.attr("data-nodekey");

                    if (!nodeElement.hasClass("disable-select"))
                        this._selectNode(self._getNodeData(nodeKey));
                    event.stopPropagation();
                },
                "click li.node": function (event) {
                    var element = $(event.currentTarget);
                    var nodeKey = element.attr("data-nodekey");

                    if (!self.options.allowNodeHighlight && self.options.allowNodeSelect && !element.hasClass("disable-select"))
                        this._selectNode(self._getNodeData(nodeKey));
                    else
                        this._highlightNode(self._getNodeData(nodeKey), event.shiftKey);
                },
                "click .h-actionButtons .h-actionButton": function (event) {
                    var element = $(event.currentTarget);
                    var nodeElement = element.closest("li.node");
                    var key = nodeElement.attr("data-nodekey");
                    var node = self._getNodeData(key);
                    var action = element.attr("data-action");

                    this._trigger("actionButtonClick", null, { node: node, action: action });
                    event.stopPropagation();
                },
                "click .h-searchButton": function (event) {
                    self._search($(".h-searchBox", self.element).val());
                },
                "keypress .h-searchBox": function (event) {
                    if (event.keyCode == 13) {
                        self._search($(".h-searchBox", self.element).val());
                    }

                    return event.keyCode != 13;
                }
            });
        },
        _getLocalizableText: function (key) {
            var text = undefined;

            try {
                if (this.options.localizableTexts[key]) {
                    text = this.options.localizableTexts[key];
                }
                else if (sharedResources && sharedResources[key]) {
                    text = sharedResources[key];
                }
                else if (resources && resources[key]) {
                    text = resources[key];
                }
            }
            catch (err) { }

            return text;
        },
        _adjustHeight: function () {
            if (this.options.height == "full") {
                var listView = $(".list-view", this.element);
                var content = $(".list-view-content", this.element);
                var dialog = $(this.element).closest(".dialog-main");
                var wrapperMarginBottom = listView.css("margin-bottom").replace("px", "");
                var height;

                if (dialog.length > 0) {
                    var contentHeight = $(".content", dialog).outerHeight(true);
                    height = contentHeight - content.position().top - wrapperMarginBottom;
                }
                else {
                    height = $(window).height() - content.offset().top - wrapperMarginBottom;
                }

                content.outerHeight(height);
            }
        }
    });
}(jQuery));